/**
 * @returns {HTMLMediaElement}
 */
const getVideo = () => {
    const video = document.querySelector('#player-container video');
    return video;
};

/**
 * 
 * @param {MediaItem} mediaItem 
 * @param {number} index 
 */
const playMediaItem = (mediaItem, index) => {
    const video = getVideo();
    video.open(mediaItem);
    document.querySelector(`#playlist tr td.selected`)?.classList.remove('selected');
    document.querySelector(`#playlist tr td[data-index="${index}"]`)?.classList.add('selected');
    // const dialog = new ConfirmDialog('이어서 재생하시겠습니까?');
    // dialog.showModal().then((returnValue) => {
    //     const video = getVideo();
    //     const startTime = returnValue === 'cancel' ? 0 : null;
    //     video.open({ ...mediaItem, startTime });
    //     document.querySelector(`#playlist tr td.selected`)?.classList.remove('selected');
    //     document.querySelector(`#playlist tr td[data-index="${index}"]`)?.classList.add('selected');
    // });
};

/**
 * @param {Playlist} playlist 
 */
const refreshPlaylist = (playlist) => {
    const table = /** @type {HTMLElement} */(document.querySelector('#playlist'));
    table.replaceChildren(...playlist.mediaItems.map((mediaItem, i) => {
        const tr = document.createElement('tr');
        const td = document.createElement('td');
        const url = new URL(mediaItem.url);
        const fileName = decodeURIComponent(url.pathname.split('/').pop());
        td.textContent = fileName;
        td.title = fileName;
        td.dataset.index = i;
        td.onclick = () => {
            playlist.moveToIndex(i, playMediaItem);
        };
        tr.append(td);
        return tr;
    }));
};

const toggleColumnRight = () => {
    const columnRight = document.querySelector('.column-right');
    if (columnRight.clientWidth === 0) {
        const main = document.querySelector('.main');
        main.style.gridTemplateColumns = '1fr 10px 250px';
    } else {
        const main = document.querySelector('.main');
        main.style.gridTemplateColumns = '1fr 10px 0px';
    }
};

/**
 * @param {HTMLMediaElement} video 
 * @param {number} index
 * @param {number} pos 
 */
const addBookmark = async (title, pos) => {
    const video = getVideo();
    const bookmarks = await video.getBookmarks();
    bookmarks.add({ title, pos });
    _refreshBookmarks(bookmarks.items);
};

/**
 * @param {HTMLMediaElement} video 
 * @param {number} index
 * @param {string} title
 */
const updateBookmark = async (index, title) => {
    const video = getVideo();
    const bookmarks = await video.getBookmarks();
    bookmarks.updateTitle(index, title);
    _refreshBookmarks(bookmarks.items);
};

/**
 * @param {number} index
 */
const removeBookmark = async (index) => {
    const video = getVideo();
    const bookmarks = await video.getBookmarks();
    bookmarks.removeAt(index);
    _refreshBookmarks(bookmarks.items);
};

const refreshBookmarks = async () => {
    if (document.querySelector('#tab-bookmark:checked~#tab-content-bookmark') == null) {
        return;
    }
    const video = getVideo();
    const bookmarks = await video.getBookmarks();
    _refreshBookmarks(bookmarks.items);
};

const _refreshBookmarks = (bookmarkItems) => {
    const bookmarkList = document.querySelector('#bookmark-list');
    bookmarkList.replaceChildren(...bookmarkItems.map((item, index) =>
        _createBookmarkItemElement(index, item))
    );
};

/**
 * @param {number} index
 * @param {BookmarkItem} bookmarkItem 
 */
const _createBookmarkItemElement = (index, bookmarkItem) => {
    const bookmarkItemElement = document.createElement('a');
    bookmarkItemElement.classList.add('bookmark-item');
    bookmarkItemElement.innerHTML = `
        <div class="bookmark-item-content">
            <div id="bookmark-item-title" class="bookmark-item-title">${bookmarkItem.title ? bookmarkItem.title : 'Untitled'}</div>
            <div class="bookmark-item-time">${_formatTime(bookmarkItem.pos)}</div>
        </div>
        <div class="bookmark-item-button-container">
            <div id="bookmark-item-edit-button" class="button-edit"></div>
            <div id="bookmark-item-delete-button" class="button-delete"></div>
        </div>
        `;
    bookmarkItemElement.onclick = () => {
        const video = getVideo();
        video.currentTime = bookmarkItem.pos;
    };
    const editButton = /** @type {HTMLElement}*/(bookmarkItemElement.querySelector('#bookmark-item-edit-button'));
    editButton.onclick = (event) => {
        event.stopPropagation();
        const titleElement = /** @type {HTMLElement}*/(bookmarkItemElement.querySelector('#bookmark-item-title'));
        titleElement.contentEditable = 'plaintext-only';
        let selection = window.getSelection();
        try {
            selection?.collapse(titleElement, 1);
        } catch (_) {
        }
        titleElement.focus();
        titleElement.onblur = () => {
            leaveEditing(true);
        };
        /** @param {KeyboardEvent} event */
        titleElement.onkeydown = (event) => {
            event.stopPropagation();
            if (event.code === 'Enter') {
                leaveEditing(true);
            } else if (event.code === 'Escape') {
                leaveEditing(false);
            }
        };
        /** @param {boolean} done */
        const leaveEditing = (done) => {
            if (done) {
                updateBookmark(index, titleElement.innerText.trim());
            }
            refreshBookmarks();
            titleElement.contentEditable = 'false';
        };
    };
    const deleteButton = bookmarkItemElement.querySelector('#bookmark-item-delete-button');
    /** @param {Event} event */
    deleteButton.onclick = (event) => {
        event.stopPropagation();
        removeBookmark(index);
    };
    return bookmarkItemElement;
};

/**
 * @param {number} time 
 */
const _formatTime = (time) => {
    const s = (time | 0) % 60;
    const m = ((time / 60) | 0) % 60;
    const h = (time / 60 / 60) | 0;
    const format = (v) => (v < 10 ? `0${v}` : `${v}`);
    return [format(h), format(m), format(s)].join(':');
};
